/**
 * Example of use of tauola C++ interface. Pythia events are
 * generated with a stable tau. Taus are subsequently decay via
 * tauola.
 *
 * @author Nadia Davidson
 * @date 17 June 2008
 */

#include "Log.h"
#include "Plots.h"
#include "Tauola.h"
#include "TauolaHepMCEvent.h"

//pythia header files
#include "Pythia.h"
#include "HepMCInterface.h"

//MC-TESTER header files
#include "Generate.h"
#include "HepMCEvent.H"
#include "Setup.H"

using namespace std;
using namespace Pythia8; 

bool ShowersOn=false;
int NumberOfEvents = 10000;

void redMinus(TauolaParticle *minus)
{
  //   
  // this method can be used to redefine branching ratios in decay of tau-
  // either generally, or specific to  tau- with pointer *minus.
  //
  // Pointer *minus can be used to define properties of decays for taus
  // at specific point(s) in the event tree. Example: 
  // vector<TauolaParticle*> x=minus->getMothers();
  // and define special versions depending on x. 
  //
  // Any combination of methods
  // Tauola::setTauBr(int mode, double br);
  // Tauola::setTaukle(double bra1, double brk0, double brk0b,double brks);
  // can be called here 
}

void redPlus(TauolaParticle *plus)
{
  //   
  // this method can be used to redefine branching ratios in decay of tau+
  // either generally, or specific to  tau+ with pointer *plus.
  //
  // Pointer *plus can be used to define properties of decays for tau
  // at specific point(s) in the event tree. Example: 
  // vector<TauolaParticle*> x=plus->getMothers();
  // and define special versions depending on x. 
  //
  // Any combination of methods
  // Tauola::setTauBr(int mode, double br);
  // Tauola::setTaukle(double bra1, double brk0, double brk0b,double brks);
  // can be called here 
}

int main(int argc,char **argv){

  Log::SummaryAtExit();
  // Initialisation of pythia
  HepMC::I_Pythia8 ToHepMC;
  Pythia pythia;
  Event& event = pythia.event;

  /*
  //Ploting class test.
  //Set incoming pdgid and costheta values for plots 1 and 2
  //Remember that table11-11.txt must be set to appropriate
  //born-level table for selected incoming pdgid plot.
  Plots::setSancVariables(1,-0.2);
  Plots::addSancPlot(1);
  Plots::addSancPlot(2);
  //Plots::addSancPlot(3); //longer
  //Plots::addSancPlot(4); //medium
  //Uncomment to quit after computing plot data
  //Plots::exitAfterSancPlots(true);
  */
  if(argc>3) NumberOfEvents=atoi(argv[3]);

  if(argc>2) ShowersOn=atoi(argv[2]);
  if(!ShowersOn)
  {
    //pythia.readString("HadronLevel:all = off");
    pythia.readString("HadronLevel:Hadronize = off");
    pythia.readString("SpaceShower:QEDshower = off");
    pythia.readString("SpaceShower:QEDshowerByL = off");
    pythia.readString("SpaceShower:QEDshowerByQ = off");
  }
  pythia.readString("PartonLevel:ISR = off");
  pythia.readString("PartonLevel:FSR = off");
  //Tauola is currently set to undecay taus. Otherwise, uncomment this line.
  //pythia.particleData.readString("15:mayDecay = off");
  if(argc>1)  //pre-set configs
  {
    pythia.readFile(argv[1]);
    pythia.init( 11, -11, 500); //e+ e- collisions
    //    pythia.init( -2212, -2212, 14000.0); //proton proton collisions
  }
  else        //default config
  {
    pythia.readString("WeakSingleBoson:ffbar2gmZ = on");
    pythia.readString("23:onMode = off"); 
    pythia.readString("23:onIfAny = 15");
    //pythia.readString("23:onIfMatch = 15 -15");
    pythia.init( 11, -11, 92.);          //electron positron collisions
    //   Tauola::setSameParticleDecayMode(19);     //19 and 22 contains K0 
    //   Tauola::setOppositeParticleDecayMode(19); // 20 contains eta
  }

  //Set up TAUOLA
  if(argc>4){
    Tauola::setSameParticleDecayMode(atoi(argv[4]));
    Tauola::setOppositeParticleDecayMode(atoi(argv[4]));
  }
  if(argc>5){
    Tauola::setHiggsScalarPseudoscalarMixingAngle(atof(argv[5]));
    Tauola::setHiggsScalarPseudoscalarPDG(25);
  }
  Tauola::initialise();
  // our default is GEV and MM, that will be outcome  units after TAUOLA
  // if HepMC unit variables  are correctly set. 
  // with the following coice you can fix the units for final outcome:
  //  Tauola::setUnits(Tauola::GEV,Tauola::MM); 
  //  Tauola::setUnits(Tauola::MEV,Tauola::CM); 
  //Tauola::setEtaK0sPi(0,0,0); // switches to decay eta K0_S and pi0 1/0 on/off. 
  //Tauola::setTauLifetime(0.0); //new tau lifetime in mm
  //Tauola::spin_correlation.setAll(false);
  //  Log::LogDebug(true);

  //Tauola::setRedefineTauMinus(redMinus);  // activates execution of routine redMinus in TAUOLA interface
  //Tauola::setRedefineTauPlus(redPlus);    // activates execution of routine redPlus  in TAUOLA interface
  MC_Initialize();
  // Begin event loop. Generate event.
  for (int iEvent = 0; iEvent < NumberOfEvents; ++iEvent) {
    if(iEvent%1000==0) Log::Info()<<"Event: "<<iEvent<<endl;
    if (!pythia.next()) continue;

    // Convert event record to HepMC
    HepMC::GenEvent * HepMCEvt = new HepMC::GenEvent();

    //Conversion needed if HepMC use different momentum units
    //than Pythia. However, requires HepMC 2.04 or higher.
    HepMCEvt->use_units(HepMC::Units::GEV,HepMC::Units::MM);

    ToHepMC.fill_next_event(event, HepMCEvt);

    //run TAUOLA on the event
    TauolaHepMCEvent * t_event = new TauolaHepMCEvent(HepMCEvt);
    //Since we let Pythia decay taus, we have to undecay them first.
    t_event->undecayTaus();
    t_event->decayTaus();
    delete t_event; 
    Log::Debug(5)<<"helicites =  "<<Tauola::getHelPlus()<<" "<<Tauola::getHelMinus()
                <<" electroweak wt= "<<Tauola::getEWwt()<<endl;

    //run MC-TESTER on the event
    HepMCEvent temp_event(*HepMCEvt,false);
    MC_Analyze(&temp_event);

    //print some events at the end of the run
    if(iEvent>=NumberOfEvents-5){  
      Log::RedirectOutput(Log::Info());
      pythia.event.list();
      HepMCEvt->print();
      Log::RevertOutput();
    }
    
    //clean up HepMC event
    delete HepMCEvt;  
  }
  Log::RedirectOutput(Log::Info());
  pythia.statistics();
  Log::RevertOutput();
  MC_Finalize();
}

