/**
 * @author T. Przedzinski
 * @date 7 Jul 2011 22:30 pm
 */

// HepMC IO_GenEvent header
#include "HepMC/IO_GenEvent.h"

// TAUOLA header
#include "Tauola/Tauola.h"

#include "TauSpinner/Particle.h"
#include "TauSpinner/SimpleParticle.h"
#include "TauSpinner/tau_reweight_lib.h"
#include "TauSpinner/nonSM.h"

#include "eventReader.h"

// LHAPDF header
#include "LHAPDF/LHAPDF.h"
using namespace std;
using namespace Tauolapp;
using namespace TauSpinner;

//ROOT headers
#include "TH1.h"   
#include "TFile.h"   

// This is an example of the program reweighting unpolarized tau events.
int main(int argc, char **argv) {

  char *input_filename = "events.dat";
  
  if(argc<2)
  {
    cout<<endl<<"Processing all available events from default input file: "<<input_filename<<endl;
    cout<<      "You can change this option by using:   "<<argv[0]<<" <filename> [<events_limit>]"<<endl<<endl;
  }
  else input_filename = argv[1];

  //---------------------------------------------------------------------------
  //- Initialization ----------------------------------------------------------
  //---------------------------------------------------------------------------

  // Limit number of processed events
  int events_limit = 0;
  
  if(argc>2) events_limit = atoi(argv[2]);
  
  // Initialize Tauola
  Tauola::initialize();

  string name="MSTW2008nnlo90cl.LHgrid";
  LHAPDF::initPDFSetByName(name);

  double CMSENE = 13000.0; // center of mass system energy.
                           // used in PDF calculation. For pp collisions only
  bool Ipp = true;         // for pp collisions
  int Ipol = 1;            // are input samples polarized?
  int nonSM2 = 0;          // are we using nonSM calculations?
                           // at present we have checked only that for nonSM2 = 0
                           // all works as in the past. nonSM2 = 1 may simply mean 
                           // errors are switched on. More work is needed 
                           // for this option.
  int nonSMN = 0;          // If we are using nonSM calculations we may want corrections 
                           // to shapes only: y/n  (1/0)

  
  // Initialize TauSpinner
  initialize_spinner(Ipp, Ipol, nonSM2, nonSMN,  CMSENE);

  // Open I/O files  (in our example events are taken from "events.dat")
  HepMC::IO_GenEvent input_file(input_filename,std::ios::in);

  int    events_count = 0;
  double wt_average   = 0.0;

  // create histograms
  TH1D *hist100 = new TH1D("hist100"," wtspin weight",100,0.0,10.0);
  TH1D *hist103 = new TH1D("hist103"," tau helicity  ",5,-2.0,2.0);
  TH1D *hist203 = new TH1D("hist203"," tau helicity, (wtspin)  ",5,-2.0,2.0);
  TH1D *hist303 = new TH1D("hist303"," tau helicity, (1/wtspin)  ",5,-2.0,2.0);

  //---------------------------------------------------------------------------
  //- Event loop --------------------------------------------------------------
  //---------------------------------------------------------------------------
  while(true) {
    double WT1    = 1.0, WT2    = 1.0;  // we assume that there may be be at most two bosons decaying into
    int    pdgid1 = 0,   pdgid2 = 0;    // taus (tau neutrinos) and requiring reweight

    SimpleParticle X, tau, tau2;      // SimpleParticle consist of 4 momentum and PDGid.
    vector<SimpleParticle> tau_daughters, tau_daughters2;

    // Here, we read another event from input_file.
    
    // In this example, we use event generated by Pythia8 with tau decays from
    // Pythia8 or Tauola++.

    // NOTE: for W+/- or H+/- tau2 contain neutrino and tau_daughters2 is empty
    // User may introduce his own method of initializing X, tau, tau2, tau_daughters, tau_daughters2.
    // Then it may be convenient to follow format of the
    // 'read_HepMC' example line below
    int status = read_HepMC(input_file, X, tau, tau2, tau_daughters, tau_daughters2);

    // Go to next one if there is nothing more to do with this event
    if( status==1 ) continue;
    
    // Finish if there is nothing more to read from the file
    if( status==0 ) break;
    
    pdgid1 = X.pdgid();

    // Sets polarization state of the input sample
    //setSpinOfSample(0);
        
    // Calculate weight for first boson
    if( abs(X.pdgid())==24 ||  abs(X.pdgid())==37 )
    {
      WT1 = calculateWeightFromParticlesWorHpn(X, tau, tau2, tau_daughters); // note that tau2 is tau neutrino
    }
    else if( X.pdgid()==25 || X.pdgid()==36 || X.pdgid()==22 || X.pdgid()==23 )
    {
      // NOTE: if any component of transverse density matrix for DY has been turned on
      //       using setZgamMultipliersTR, weight WT will contain transverse part
      //       of the spin amplitude (calculated from tables table1-1.txt table2-2.txt
      //       which must be stored in the same directory as the program)
      WT1 = calculateWeightFromParticlesH(X, tau, tau2, tau_daughters,tau_daughters2);
    }
    else
    {
      cout<<"WARNING: Unexpected PDG for tau mother: "<<X.pdgid()<<endl;
    }
    
    // Returns spin of the taus attributed during reweighting
    double pol = getTauSpin();
    
    //--------------------------------------------------------------------------
    // If event was not finished processing, then status = 2.
    // In this case, check if there is a second boson
    // decaying into tau in the event
    //--------------------------------------------------------------------------
    if( status==2 )
    {
      int status2 = read_HepMC(input_file, X, tau, tau2, tau_daughters, tau_daughters2);

      if( status2==2 )
      {
        pdgid2 = X.pdgid();

        // Calculate second weight
        if( abs(X.pdgid())==24 ||  abs(X.pdgid())==37 )
        {
          WT2 = calculateWeightFromParticlesWorHpn(X, tau, tau2, tau_daughters); // note that tau2 is tau neutrino
        }
        else if( X.pdgid()==25 || X.pdgid()==36 || X.pdgid()==22 || X.pdgid()==23 )
        {
          WT2 = calculateWeightFromParticlesH(X, tau, tau2, tau_daughters,tau_daughters2);
        }
        else
        {
          cout<<"WARNING: Unexpected PDG for tau mother: "<<X.pdgid()<<endl;
        }
        
      } // if( status2==2 )
    } // if( status==2 )
    
    // At this step we have two weights: WT1 and WT2.
    // If there was no 2nd boson decaying to taus, WT2 = 1.0
    
    // Options:
    // WT=1.0/WT      // to unspin sample having spin
    // WT=(2-WT)/(WT) // to flip spin correlations between (H+- <-->  W+-)
    // WT=(2-WT)/(WT) // to flip spin correlations between (H <-->  Z/gamma)
    //                   (this simple trick is valid with no Z couplings taken)
    //                   for Z calculation of weight for Higgs has to be done
    //                   directly
    // WARNING: these new WT will not be limited from above!
    // Run MC-TESTER on the event
    // Use WT to check if spin is properly installed (unweight events).

    //here select first resonance in the event
    double WT = WT1;
    
    wt_average += WT;
    events_count++;

    hist100->Fill(WT);
    hist103->Fill(pol,1.0);
    hist203->Fill(pol,WT);
    hist303->Fill(pol,1/WT);

    if(events_count%10000==0) cout<<"EVT: "<<events_count<<endl;
    if(events_limit && events_count>=events_limit) break;
  }
    wt_average = wt_average/events_count;

  cout<<"No of events processed: "<<events_count<<endl;
  cout<<"WT average for these events: "<<wt_average<<endl;

  char output_filename[500]="";
  strcat (output_filename,input_filename);
  strcat (output_filename,".root");
  TFile outFile(output_filename, "recreate");

  outFile.cd();
  hist100->Write();
  hist103->Write();
  hist203->Write();
  hist303->Write();
  outFile.Close();



}
