#!/bin/bash

set -e

##############################################################################
# Configuration                                                              #
##############################################################################

BASE_INSTALL_DIR=$PWD

HEPMC2_LINK="https://gitlab.cern.ch/hepmc/HepMC/-/archive/HEPMC_02_06_11/HepMC-HEPMC_02_06_11.tar"
HEPMC2_EXPECTED_DIR="HepMC-HEPMC_02_06_11"

HEPMC3_LINK="https://gitlab.cern.ch/hepmc/HepMC3/-/archive/3.3.0/HepMC3-3.3.0.tar"
HEPMC3_EXPECTED_DIR="HepMC3-3.3.0"

LHAPDF_LINK="https://lhapdf.hepforge.org/downloads/?f=LHAPDF-6.5.4.tar.gz"
LHAPDF_EXPECTED_DIR="LHAPDF-6.5.4"

PYTHIA_LINK="https://pythia.org/download/pythia83/pythia8312.tgz"
PYTHIA_EXPECTED_DIR="pythia8312"

MC_TESTER_LINK="https://gitlab.cern.ch/cvsmctst/mc-tester/-/archive/master/mc-tester-master.tar"
MC_TESTER_EXPECTED_DIR="mc-tester-master"

TAUOLA_LINK="https://tauolapp.web.cern.ch/tauolapp/resources/TAUOLA.development.version/TAUOLA.development.version.tar.gz"
TAUOLA_EXPECTED_DIR="TAUOLA"

PHOTOS_LINK="https://photospp.web.cern.ch/photospp/resources/PHOTOS.3.61/PHOTOS.3.61.tar.gz"
PHOTOS_EXPECTED_DIR="PHOTOS"

LHAPDF_DATASET_LINK="https://lhapdfsets.web.cern.ch/current/MSTW2008nnlo90cl.tar.gz"
##############################################################################
# Install HepMC2                                                             #
##############################################################################

install_hepmc2() {
  wget $HEPMC2_LINK
  ARCHIVE=$(basename $HEPMC2_LINK)
  tar -xvf $ARCHIVE
  rm $ARCHIVE
  mv $HEPMC2_EXPECTED_DIR HepMC2-src
  cd HepMC2-src/
  mkdir build
  cd build/
  cmake .. -DCMAKE_INSTALL_PREFIX=$BASE_INSTALL_DIR/HepMC2 -Dmomentum=GEV -Dlength=MM
  make -j
  make install -j

  cd $BASE_INSTALL_DIR
  rm -rf HepMC2-src
}

##############################################################################
# Install HepMC3                                                             #
##############################################################################

install_hepmc3() {
  wget $HEPMC3_LINK
  ARCHIVE=$(basename $HEPMC3_LINK)
  tar -xvf $ARCHIVE
  rm $ARCHIVE
  mv $HEPMC3_EXPECTED_DIR HepMC3-src
  cd HepMC3-src/
  mkdir build
  cd build/
  cmake .. -DCMAKE_INSTALL_PREFIX=$BASE_INSTALL_DIR/HepMC3 -DHEPMC3_ENABLE_PYTHON=OFF
  make -j
  make install -j
  cd ../..
  rm -rf HepMC3-src

  # FIXME: Fix the HepMC3 installation
  cd $BASE_INSTALL_DIR/HepMC3
  ln -s lib64 lib

  cd $BASE_INSTALL_DIR
}

##############################################################################
# Install LHAPDF                                                             #
##############################################################################
install_lhapdf() {
  wget $LHAPDF_LINK -O lhapdf.tgz
  ARCHIVE=lhapdf.tgz
  tar -xvzf "$ARCHIVE"
  rm $ARCHIVE
  mv $LHAPDF_EXPECTED_DIR lhapdf-src
  cd lhapdf-src/
  ./configure --prefix=$BASE_INSTALL_DIR/lhapdf --disable-python
  make -j
  make install

  cd $BASE_INSTALL_DIR
  rm -rf lhapdf-src
}

##############################################################################
# Install Pythia                                                             #
##############################################################################

install_pythia() {
  wget $PYTHIA_LINK
  ARCHIVE=$(basename $PYTHIA_LINK)
  tar -xvzf $ARCHIVE
  rm $ARCHIVE
  mv $PYTHIA_EXPECTED_DIR pythia
  cd pythia/
  ./configure --with-hepmc2=$BASE_INSTALL_DIR/HepMC2 --with-hepmc3=$BASE_INSTALL_DIR/HepMC3
  make -j 8

  cd $BASE_INSTALL_DIR
}

##############################################################################
# Install MC-TESTER                                                          #
##############################################################################

install_mc_tester() {
  wget $MC_TESTER_LINK
  ARCHIVE=$(basename $MC_TESTER_LINK)
  tar -xvf $ARCHIVE
  rm $ARCHIVE
  mv $MC_TESTER_EXPECTED_DIR mc-tester
  cd mc-tester
  ./configure --with-HepMC=$BASE_INSTALL_DIR/HepMC2 --with-HepMC3=$BASE_INSTALL_DIR/HepMC3
  make -j

  cd $BASE_INSTALL_DIR
}

##############################################################################
# Install TAUOLA                                                             #
##############################################################################

install_tauola() {
  wget $TAUOLA_LINK
  ARCHIVE=$(basename $TAUOLA_LINK)
  tar -xvzf $ARCHIVE
  rm $ARCHIVE
  mv $TAUOLA_EXPECTED_DIR tauola
  cd tauola

  # FIXME: This is a hack to make the build work, because currently f77 is detected instead of gfortran
  sed -i 's/@F77@/gfortran/g' platform/make.inc.in

  ./configure \
      --with-hepmc=$BASE_INSTALL_DIR/HepMC2 \
      --with-hepmc3=$BASE_INSTALL_DIR/HepMC3 \
      --with-lhapdf=$BASE_INSTALL_DIR/lhapdf \
      --with-mc-tester=$BASE_INSTALL_DIR/mc-tester \
      --with-pythia8=$BASE_INSTALL_DIR/pythia \
      --with-tau-spinner
  make

  # Examples
  cd examples/
  ./configure
  make

  cd ../TauSpinner/examples/
  ./configure \
      --with-hepmc=$BASE_INSTALL_DIR/HepMC2 \
      --with-mc-tester=$BASE_INSTALL_DIR/mc-tester
  make

  cd $BASE_INSTALL_DIR
}


##############################################################################
# Install PHOTOS                                                             #
##############################################################################

install_photos() {
  wget $PHOTOS_LINK
  ARCHIVE=$(basename $PHOTOS_LINK)
  tar -xvzf $ARCHIVE
  rm $ARCHIVE
  mv $PHOTOS_EXPECTED_DIR photos
  cd photos

  # FIXME: This is a hack to make the build work, because currently f77 is detected instead of gfortran
  sed -i 's/@F77@/gfortran/g' platform/make.inc.in

  ./configure \
      --with-hepmc=$BASE_INSTALL_DIR/HepMC2 \
      --with-hepmc3=$BASE_INSTALL_DIR/HepMC3 \
      --with-mc-tester=$BASE_INSTALL_DIR/mc-tester \
      --with-pythia8=$BASE_INSTALL_DIR/pythia \
      --with-tauola=$BASE_INSTALL_DIR/TAUOLA

  make -j

  # Examples
  cd examples/
  ./configure
  make

  cd $BASE_INSTALL_DIR
}

##############################################################################
# Install LHAPDF dataset                                                     #
##############################################################################

install_lhapdf_dataset() {
  cd $BASE_INSTALL_DIR/lhapdf/share/LHAPDF/
  wget $LHAPDF_DATASET_LINK
  ARCHIVE=$(basename $LHAPDF_DATASET_LINK)
  tar -xvzf $ARCHIVE
  rm $ARCHIVE

  cd $BASE_INSTALL_DIR/tauola/TauSpinner/examples
  ./tau-reweight-test.exe

  cd $BASE_INSTALL_DIR
}

##############################################################################
# Main                                                                       #
##############################################################################

install_hepmc2
install_hepmc3
install_lhapdf
install_pythia
install_mc_tester
install_tauola
install_photos
install_lhapdf_dataset

echo ""
echo "Done. Consider:"
echo "  source photos/configure.paths.sh"
