/**
 * @author T. Przedzinski
 * @date 7 Jul 2011 22:30 pm
 */

// HepMC IO_GenEvent header
#include "HepMC/IO_GenEvent.h"

// TAUOLA header
#include "Tauola/Tauola.h"

#include "TauSpinner/Particle.h"
#include "TauSpinner/SimpleParticle.h"
#include "TauSpinner/tau_reweight_lib.h"
#include "TauSpinner/nonSM.h"

#include "eventReader.h"

// LHAPDF header
#include "LHAPDF/LHAPDF.h"
using namespace std;
using namespace Tauolapp;
using namespace TauSpinner;

//ROOT headers
#include "TH1.h"   
#include "TFile.h"  


/* Get decay channel
   Returns:
   -1 - unrecognized (unsupported) channel
    0 - no daughters
    1 - electron (tau -> nu_tau nue e [+gamma])
    2 - muon     (tau -> nu_tau nu_mu mu [+gamma])
    3 - pion     (tau -> nu_tau pi)
    4 - rho      (tau -> nu_tau pi0 pi) */
int get_decay_channel(vector<SimpleParticle> &tau_daughters);
 

// This is an example of the program reweighting unpolarized tau events.
int main(int argc, char **argv) {

  char *input_filename = "events.dat";
  
  if(argc<2)
  {
    cout<<endl<<"Processing all available events from default input file: "<<input_filename<<endl;
    cout<<      "You can change this option by using:   "<<argv[0]<<" <filename> [<events_limit>]"<<endl<<endl;
  }
  else input_filename = argv[1];

  //---------------------------------------------------------------------------
  //- Initialization ----------------------------------------------------------
  //---------------------------------------------------------------------------

  // Limit number of processed events
  int events_limit = 0;
  
  if(argc>2) events_limit = atoi(argv[2]);
  
  // Initialize Tauola
  Tauola::initialize();

  string name="MSTW2008nnlo90cl.LHgrid";
  LHAPDF::initPDFSetByName(name);

  double CMSENE = 13000.0; // center of mass system energy.
                           // used in PDF calculation. For pp collisions only
  bool Ipp = true;         // for pp collisions
  int Ipol = 1;            // are input samples polarized?
  int nonSM2 = 0;          // are we using nonSM calculations?
                           // at present we have checked only that for nonSM2 = 0
                           // all works as in the past. nonSM2 = 1 may simply mean 
                           // errors are switched on. More work is needed 
                           // for this option.
  int nonSMN = 0;          // If we are using nonSM calculations we may want corrections 
                           // to shapes only: y/n  (1/0)

  
  // Initialize TauSpinner
  initialize_spinner(Ipp, Ipol, nonSM2, nonSMN,  CMSENE);

  // Open I/O files  (in our example events are taken from "events.dat")
  HepMC::IO_GenEvent input_file(input_filename,std::ios::in);

  int    events_count = 0;
  double wt_average   = 0.0;

  // create histograms
  TH1D *hist100 = new TH1D("hist100"," wtspin weight",25,0.0,10.0);
  TH1D *hist101 = new TH1D("hist101"," Energy fraction pi+-",25,0.0,1.0);
  TH1D *hist102 = new TH1D("hist102"," Energy fraction rho ",25,0.0,1.0);
  TH1D *hist103 = new TH1D("hist103"," tau helicity  ",5,-2.0,2.0);
  TH1D *hist201 = new TH1D("hist201"," Energy fraction pi+-, (wtspin)",25,0.0,1.0);
  TH1D *hist202 = new TH1D("hist202"," Energy fraction rho,  (wtspin)",25,0.0,1.0);
  TH1D *hist203 = new TH1D("hist203"," tau helicity  ",5,-2.0,2.0);
  TH1D *hist301 = new TH1D("hist301"," Energy fraction pi+-, (1./wtspin)",25,0.0,1.0);
  TH1D *hist302 = new TH1D("hist302"," Energy fraction rho,  (1./wtspin)",25,0.0,1.0);
  TH1D *hist303 = new TH1D("hist303"," tau helicity  ",5,-2.0,2.0);

  //---------------------------------------------------------------------------
  //- Event loop --------------------------------------------------------------
  //---------------------------------------------------------------------------
  while(true) {
    double WT    = 1.0;  // we assume that there is only one bosons decaying into

    SimpleParticle X, tau1, tau2;      // SimpleParticle consist of 4 momentum and PDGid.
    vector<SimpleParticle> tau1_daughters, tau2_daughters;

    // Here, we read another event from input_file.
    
    // In this example, we use event generated by Pythia8 with tau decays from
    // Pythia8 or Tauola++.

    // NOTE: for W+/- or H+/- tau2 contain neutrino and tau_daughters2 is empty
    // User may introduce his own method of initializing X, tau, tau2, tau_daughters, tau_daughters2.
    // Then it may be convenient to follow format of the
    // 'read_HepMC' example line below
    int status = read_HepMC(input_file, X, tau1, tau2, tau1_daughters, tau2_daughters);

    // Go to next one if there is nothing more to do with this event
    if( status==1 ) continue;
    
    // Finish if there is nothing more to read from the file
    if( status==0 ) break;
    
    // Sets polarization state of the input sample
    //setSpinOfSample(0);
        
    // Calculate weight for first boson
    if( abs(X.pdgid())==24 ||  abs(X.pdgid())==37 )
    {
      WT = calculateWeightFromParticlesWorHpn(X, tau1, tau2, tau1_daughters); // note that tau2 is tau neutrino
    }
    else if( X.pdgid()==25 || X.pdgid()==36 || X.pdgid()==22 || X.pdgid()==23 )
    {
      // NOTE: if any component of transverse density matrix for DY has been turned on
      //       using setZgamMultipliersTR, weight WT will contain transverse part
      //       of the spin amplitude (calculated from tables table1-1.txt table2-2.txt
      //       which must be stored in the same directory as the program)
      WT = calculateWeightFromParticlesH(X, tau1, tau2, tau1_daughters,tau2_daughters);
    }
    else
    {
      cout<<"WARNING: Unexpected PDG for tau mother: "<<X.pdgid()<<endl;
    }
    
    // Returns spin of the taus attributed during reweighting
    double pol = getTauSpin();
    
    //--------------------------------------------------------------------------
    // If event was not finished processing, then status = 2.
    // However skip the rest of the event
    
    wt_average += WT;
    events_count++;

    // base weight (not related to TauSpinner)
    double weight = 1.0;

    // get decay channels of the daughters
    int channel1 = get_decay_channel(tau1_daughters);
    int channel2 = get_decay_channel(tau2_daughters);
      
    // Analyze only events that belong to supported decay channels
    if(channel1>0) {
      // Energy spectra of charged pion/lepton
      double e1         = tau1_daughters[1].e()/tau1.e();
 
      if(channel1==3) {
	hist101->Fill(e1,weight);    
	hist201->Fill(e1,weight*WT);    
	hist301->Fill(e1,weight/WT);    
      }      
      // For rho - if no rho is explicitly written in the event record,
      // sum of the energy of pi-/+ and pi0 is used
      if(channel1==4 && tau1_daughters.size()==3) {
	e1 = (tau1_daughters[1].e() + tau1_daughters[2].e())/tau1.e();
	hist102->Fill(e1,weight);    
	hist202->Fill(e1,weight*WT);    
	hist302->Fill(e1,weight/WT);    
      }     
    }
    if(channel2>0) {
      // Energy spectra of charged pion/lepton
      double e2         = tau2_daughters[1].e()/tau2.e();
 
      if(channel2==3) {
	hist101->Fill(e2,weight);    
	hist201->Fill(e2,weight*WT);    
	hist301->Fill(e2,weight/WT);    

      }      
      // For rho - if no rho is explicitly written in the event record,
      // sum of the energy of pi-/+ and pi0 is used
      if(channel2==4 && tau2_daughters.size()==3) {
	e2 = (tau2_daughters[1].e() + tau2_daughters[2].e())/tau2.e();
	hist102->Fill(e2,weight);    
	hist202->Fill(e2,weight*WT);    
	hist302->Fill(e2,weight/WT);    
      }     
    }

    hist100->Fill(WT);
    hist103->Fill(pol,weight);
    hist203->Fill(pol,weight*WT);
    hist303->Fill(pol,weight/WT);

    if(events_count%10000==0) cout<<"EVT: "<<events_count<<endl;
    if(events_limit && events_count>=events_limit) break;
  }


  //---------------------------------------------------------------------------
  //- Finalisation  --------------------------------------------------------------
  //---------------------------------------------------------------------------

  wt_average = wt_average/events_count;

  cout<<"No of events processed: "<<events_count<<endl;
  cout<<"WT average: "<<wt_average<<endl;

  char output_filename[500]="";
  strcat (output_filename,input_filename);
  strcat (output_filename,".root");
  TFile outFile(output_filename, "recreate");

  outFile.cd();
  hist100->Write();
  hist101->Write();
  hist102->Write();
  hist103->Write();
  hist201->Write();
  hist202->Write();
  hist203->Write();
  hist301->Write();
  hist302->Write();
  hist303->Write();
  outFile.Close();



}

/* Get decay channel
   Returns:
   -1 - unrecognized (unsupported) channel
    0 - no daughters
    1 - electron (tau -> nu_tau nue e [+gamma])
    2 - muon     (tau -> nu_tau nu_mu mu [+gamma])
    3 - pion     (tau -> nu_tau pi)
    4 - rho      (tau -> nu_tau pi0 pi) */
int get_decay_channel(vector<SimpleParticle> &tau_daughters)
{
  int size = tau_daughters.size();
  
  if(size==0) return 0;

  int id[4] = { 0 };
  if(size>0) id[0] = abs(tau_daughters[0].pdgid());
  if(size>1) id[1] = abs(tau_daughters[1].pdgid());
  if(size>2) id[2] = abs(tau_daughters[2].pdgid());
  if(size>3) id[3] = abs(tau_daughters[3].pdgid());

  // nu_tau pi
  if(size==2 && id[1]==211) return 3;
  
  // nu_tau rho
  if(size==2 && id[1]==213) return 4;
  
  if(size==3)
  {
    // nu_tau e nu_e 
    if(id[1]==11) return 1;
    
    // nu_tau mu nu_mu 
    if(id[1]==13) return 2;
    
    // nu_tau pi0 pi (without intermediate rho)
    if( (id[1]==111 && id[2]==211) ||
        (id[1]==211 && id[2]==111)    ) return 4;
  }
  
  if(size==4 && id[3]==22)
  {
    // nu tau e nu_e gamma
    if(id[1]==11) return 1;
    
    // nu tau mu nu_mu gamma
    if(id[1]==13) return 2;
  }
  
  return -1;
}
